<?php
/**
 * WooCommerce Product Sorting Widget
 * Widget for sorting WooCommerce products based on date, rating, price, popularity, etc.
 */
class SS_Product_Sorting extends WP_Widget {

	protected static $instance;

	// Register widget with WordPress.
	function __construct() {
		parent::__construct(
			'ss_product_sorting', // Base ID
			esc_attr__( 'SS Product Sorting', 'sswcaf' ), // Name
			array( 'description' => esc_attr__( 'WooCommerce ajax product sorting widget', 'sswcaf' ), ) // Args
		);
		self::$instance = false;
	}

	// Front-end display of widget.
	public function widget( $args, $instance ) {

		// Exit if it is not a WooCommerce product archive or shop
		if ( ! ( is_post_type_archive( 'product' ) || is_tax( get_object_taxonomies( 'product' ) ) ) ) {
			return;
		}

		// Exit if widget instance already exists
		if ( ! self::$instance ) {
			self::$instance = true;
		}
		else {
			return;
		}

		$options = get_option( 'sswcaf_settings' );
		if ( isset( $options ) ) {
			$enable_filtering_per_sel = isset( $options['sswcaf_enable_filtering_per_selection'] ) ? $options['sswcaf_enable_filtering_per_selection'] : false;
		}

		$hide_buttons = isset( $instance['buttons'] ) ? $instance['buttons'] : false;

		// Widget output
		echo $args['before_widget'];

		if ( ! empty( $instance['title'] ) ) {
			echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ). $args['after_title'];
		}

		// Get default sorting option as set inside Customizer
		$wc_query = new WC_Query();
		$order_args = $wc_query->get_catalog_ordering_args( $orderby = '', $order = '' );		
		if ( isset( $order_args ) && is_array( $order_args ) ) {
			$orderby = $order_args['orderby'];
		}
		
		// Check if orderby argument is set
		if ( isset( $_GET['orderby'] ) ) {
			$orderby =  $_GET['orderby'];
		}
		
		$sorting_option	=  get_option( 'woocommerce_enable_review_rating' );
		
		$choices = apply_filters( 'woocommerce_default_catalog_orderby_options', array(
			'menu_order' => __( 'Default sorting (custom ordering + name)', 'sswcaf' ),
			'popularity' => __( 'Popularity (sales)', 'sswcaf' ),
			'rating'     => __( 'Average rating', 'sswcaf' ),
			'date'       => __( 'Sort by most recent', 'sswcaf' ),
			'price'      => __( 'Sort by price (asc)', 'sswcaf' ),
			'price-desc' => __( 'Sort by price (desc)', 'sswcaf' ),
		) );

		$out = '<ul class="ss-product-sorting type-radio">';
		foreach( $choices as $key => $value ) {
			$out .= sprintf( '<li%s data-orderby="%s">%s</li>',
				$key == $orderby ? ' class="chosen"' : '',
				esc_attr( $key ),				
				esc_attr( $value )
			);
		}

		$out .= '</ul>';

		echo $out;

		if ( ! $hide_buttons && ! $enable_filtering_per_sel ) { ?>
			<p class="product-query-buttons">
				<button type="button" class="ss-btn btn-primary apply-query"><?php esc_attr_e( 'Apply', 'sswcaf' ); ?></button><button type="button" class="ss-btn btn-flat reset-sorting"><?php esc_attr_e( 'Reset', 'sswcaf' ); ?></button>
			</p>
		<?php }

		echo $args['after_widget'];
	}

	// Back-end widget form.
	public function form( $instance ) {
		$title		= ! empty( $instance['title'] ) ? $instance['title'] : esc_attr__( 'New title', 'sswcaf' );
		$instance	= wp_parse_args( (array) $instance, array( 'buttons' => false ) );
		?>
		<p>
            <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php esc_attr_e( 'Title:' ); ?></label>
            <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>">
		</p>
		<p>
            <input class="checkbox" type="checkbox" <?php checked($instance['buttons'], true) ?> id="<?php echo $this->get_field_id('buttons'); ?>" name="<?php echo $this->get_field_name('buttons'); ?>" />
            <label for="<?php echo $this->get_field_id('buttons'); ?>"><?php esc_attr_e('Hide action buttons'); ?></label>
		</p>
		<?php
	}

	// Sanitize widget form values as they are saved.
	public function update( $new_instance, $old_instance ) {
		$new_instance = (array) $new_instance;
		$instance = array();
		$instance['title'] = ( ! empty( $new_instance['title'] ) ) ? strip_tags( $new_instance['title'] ) : '';
		$instance['buttons'] = isset($new_instance['buttons']) ? true : false;
		return $instance;
	}
}