<?php
/**
 * WooCommerce Product Price Range Widget
 * Filters product results based on chosen price range
 */

class SS_Product_Price_Range extends WP_Widget {

	protected static $instance;

	// Register widget with WordPress.
	function __construct() {
		parent::__construct(
			'ss_product_price_range',
			__( 'SS Product Price Range', 'sswcaf' ),
			array( 'description' => esc_attr__( 'WooCommerce Product Price Range', 'sswcaf' ) )
		);

		self::$instance = false;
	}

	// Front-end display of widget
	public function widget( $args, $instance ) {

		global $_chosen_attributes, $wpdb, $wp;

		// Exit if it is not a WooCommerce product archive or shop
		if ( ! ( is_post_type_archive( 'product' ) || is_tax( get_object_taxonomies( 'product' ) ) ) ) {
			return;
		}

		// Exit if widget instance already exists
		if ( ! self::$instance ) {
			self::$instance = true;
		}
		else {
			return;
		}

		$hide_buttons	= isset( $instance['buttons'] ) ? $instance['buttons'] : false;
		$range			= empty( $instance['range'] ) ? '' : wp_filter_nohtml_kses( $instance['range'] );

		if ( empty ( $range ) ) {
			return;
		}

		$options = get_option( 'sswcaf_settings' );
		if ( isset( $options ) ) {
			$enable_filtering_per_sel = isset( $options['sswcaf_enable_filtering_per_selection'] ) ? $options['sswcaf_enable_filtering_per_selection'] : false;
		}

		// Widget output
		echo $args['before_widget'];

		if ( ! empty( $instance['title'] ) ) {
			echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ). $args['after_title'];
		}

		// Output
		if ( function_exists( 'get_woocommerce_currency_symbol' ) ) {
			$currency = apply_filters( 'ss_product_price_filter_currency_symbol', get_woocommerce_currency_symbol() );
		}
		else {
			$currency = apply_filters( 'ss_product_price_filter_currency_symbol', '' );
		}

		$min_price = isset( $_GET['min_price'] ) ? esc_attr( $_GET['min_price'] ) : null;
		$max_price = isset( $_GET['max_price'] ) ? esc_attr( $_GET['max_price'] ) : null;
		$chosen_class = '';

		if ( ! empty ( $range ) ) {
			$set_arr = array();
			$set = explode( '|', $range );
			if ( is_array( $set ) ) {
				foreach ( $set as $set_item ) {
					$set_arr[] = explode( ',', $set_item );
				}
			}

			$out = '<ul class="ss-product-price-range type-radio">';

			foreach ( $set_arr as $key => $value ) {
				if ( (int)$value[0] <= 0 && (int)$value[1] > 0 ) {
					$chosen_class = ( isset( $max_price ) && (int)$max_price <= (int)$value[1] ) ? ' class="chosen"' : '';
					$out .= '<li' . $chosen_class . ' data-minprice="0" data-maxprice="' . (int)$value[1] . '">' . sprintf( _x( 'Below %1$s%2$s', 'price range below xx amount', 'sswcaf' ), $currency, (int)$value[1] ) . '<span class="count">(' . sswcaf_price_count( 0, (int)$value[1] ) . ')</span></li>';
				}

				elseif ( (int)$value[0] > 0 && (int)$value[1] > (int)$value[0] ) {
					$chosen_class = ( isset( $max_price ) && isset( $min_price ) && (int)$min_price >= (int)$value[0] && (int)$max_price <= (int)$value[1] ) ? ' class="chosen"' : '';
					$out .= '<li' . $chosen_class . ' data-minprice="' . (int)$value[0] . '" data-maxprice="' . (int)$value[1] . '">' . sprintf( _x( 'From %1$s%2$s to %1$s%3$s', 'price range between xx to yy amount', 'sswcaf' ), $currency, (int)$value[0], (int)$value[1] ) . '<span class="count">(' . sswcaf_price_count( (int)$value[0], (int)$value[1] ) . ')</span></li>';
				}

				elseif ( (int)$value[0] > 0 && (int)$value[1] <= 0 ) {
					$chosen_class = ( isset( $min_price ) && (int)$min_price >= (int)$value[0] ) ? ' class="chosen"' : '';
					$out .= '<li' . $chosen_class . ' data-minprice="' . (int)$value[0] . '" data-maxprice="0">' . sprintf( _x( 'Above %1$s%2$s', 'price range above xx amount', 'sswcaf' ), $currency, (int)$value[0] )  . '<span class="count">('  . sswcaf_price_count( (int)$value[0], 999999999 ) . ')</span></li>';
				}
			}

			$out .= '</ul>';

			echo $out;

			if ( ! $hide_buttons && ! $enable_filtering_per_sel ) { ?>
				<p class="product-query-buttons">
					<button type="button" class="ss-btn btn-primary apply-query"><?php esc_attr_e( 'Apply', 'sswcaf' ); ?></button><button type="button" class="ss-btn btn-flat clear-price-range"><?php esc_attr_e( 'Clear', 'sswcaf' ); ?></button>
				</p>
			<?php }
		}

		echo $args['after_widget'];
	}


	// Back-end widget form.
	public function form( $instance ) {

		$title		= ! empty( $instance['title'] ) ? $instance['title'] : esc_attr__( 'New title', 'sswcaf' );
		$range		= ! empty( $instance['range'] ) ? esc_textarea( $instance['range'] ) : '';
		$instance	= wp_parse_args( (array) $instance, array( 'buttons' => false ) );
		?>
		<p>
            <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php esc_attr_e( 'Title:' ); ?></label>
            <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>">
		</p>
		<p>
            <label for="<?php echo $this->get_field_id( 'range' ); ?>"><?php esc_attr_e( 'Range data:' ); ?></label>
            <textarea class="widefat" rows="4" cols="20" id="<?php echo $this->get_field_id('range'); ?>" name="<?php echo $this->get_field_name('range'); ?>"><?php echo $range; ?></textarea>
            <small><?php esc_attr_e( 'Use comma separated values for each range. Separate each set with a pipe. E.g. 0,20 | 20,40 | 40,60 | 60, 0', 'sswcaf' ); ?></small>
        </p>
		<p>
            <input class="checkbox" type="checkbox" <?php checked($instance['buttons'], true) ?> id="<?php echo $this->get_field_id('buttons'); ?>" name="<?php echo $this->get_field_name('buttons'); ?>" />
            <label for="<?php echo $this->get_field_id('buttons'); ?>"><?php esc_attr_e('Hide action buttons'); ?></label>
		</p>

	<?php
    }

	// Sanitize widget form values as they are saved.
	public function update( $new_instance, $old_instance ) {
		$new_instance = (array) $new_instance;
		$instance = array();
		$instance['title'] = ( ! empty( $new_instance['title'] ) ) ? strip_tags( $new_instance['title'] ) : '';
		if ( current_user_can('unfiltered_html') ) {
			$instance['range'] =  $new_instance['range'];
		}
		else {
			$instance['range'] = stripslashes( wp_filter_post_kses( addslashes($new_instance['range']) ) );
		}
		$instance['buttons'] = isset($new_instance['buttons']) ? true : false;

		return $instance;
	}

}