<?php
/**
 * Helper functions used in the plugin
 */

/**
 * Walker_Category class extended for Attributes Widget
 *
 * Adds data-slug="xx" inside taxonomy links
 * https://codex.wordpress.org/Class_Reference/Walker_Category
 */

class SS_Prduct_Filter_Walker extends Walker_Category {

	var $db_fields = array ( 'parent' => 'parent', 'id' => 'term_id' );
	protected $count;

	/**
	 * Starts the element output.
	 *
	 * @since 2.1.0
	 *
	 * @see Walker::start_el()
	 *
	 * @param string $output   Used to append additional content (passed by reference).
	 * @param object $category Category data object.
	 * @param int    $depth    Optional. Depth of category in reference to parents. Default 0.
	 * @param array  $args     Optional. An array of arguments. See wp_list_categories(). Default empty array.
	 * @param int    $id       Optional. ID of the current category. Default 0.
	 */
	 
	function start_el( &$output, $category, $depth = 0, $args = array(), $id = 0 ) {

		global $wp_query;

		$vars = array_filter( $wp_query->query_vars );
		$vars['posts_per_page'] = -1;
		unset( $vars['term'] );
		unset( $vars['taxonomy'] );

		// Generate new query after resetting terms and taxonomy
		$new_query = new WP_Query( $vars );

		if ( ! empty( $args['walker_posts'] ) ) {
			$posts = $args['walker_posts'];
		}
		else {
			$posts = $new_query->posts;
		}

		if ( ! empty( $args['walker_terms'] ) ) {
			$terms = $args['walker_terms'];
		}
		else {
			$terms = null;
		}

		$taxonomy = $args['walker_taxonomy'];

		$delimeter = ( $args['walker_operator'] == 'AND' ) ? '+' : ',';

		$tax_terms = ( isset( $_GET[ $args['walker_taxonomy'] ] ) ) ? explode( $delimeter, $_GET[ $args['walker_taxonomy'] ] ) : array();
		$this->_count = 0;
		$this->_tax_flag = true;

		foreach( $posts as $post ) {
			if ( has_term( $category->slug, $taxonomy, $post ) ) {
				$this->_count++;
			}

			if ( is_tax() ) {
				$current_tax_term = get_queried_object()->slug;
				if ( is_tax( $taxonomy, $current_tax_term ) && $category->slug == $current_tax_term ) {
					$this->_tax_flag = false;
				}
			}
		}

		/** This filter is documented in wp-includes/category-template.php */
		$cat_name = apply_filters(
			'list_cats',
			esc_attr( $category->name ),
			$category
		);

		// Don't generate an element if the category name is empty.
		if ( ! $cat_name ) {
			return;
		}

		$link = '<a data-slug="' . $category->slug . '" data-id="' . $category->term_id . '" href="' . get_term_link( $category ) . '" ';
		if ( $args['use_desc_for_title'] && ! empty( $category->description ) ) {
			$link .= 'title="' . esc_attr( strip_tags( apply_filters( 'category_description', $category->description, $category ) ) ) . '"';
		}

		$link .= '>';
		$link .= $cat_name . '</a>';

		if ( ! empty( $args['feed_image'] ) || ! empty( $args['feed'] ) ) {
			$link .= ' ';

			if ( empty( $args['feed_image'] ) ) {
				$link .= '(';
			}

			$link .= '<a href="' . esc_url( get_term_feed_link( $category->term_id, $category->taxonomy, $args['feed_type'] ) ) . '"';

			if ( empty( $args['feed'] ) ) {
				$alt = ' alt="' . sprintf( __( 'Feed for all posts filed under %s', 'sswcaf' ), $cat_name ) . '"';
			} else {
				$alt = ' alt="' . $args['feed'] . '"';
				$name = $args['feed'];
				$link .= empty( $args['title'] ) ? '' : $args['title'];
			}

			$link .= '>';

			if ( empty( $args['feed_image'] ) ) {
				$link .= $name;
			} else {
				$link .= "<img src='" . $args['feed_image'] . "'$alt" . ' />';
			}
			$link .= '</a>';

			if ( empty( $args['feed_image'] ) ) {
				$link .= ')';
			}
		}

		if ( ! empty( $args['show_count'] ) ) {
				$link .= '<span class="count">(' . number_format_i18n( $this->_count ) . ')</span>';
		}

		// Use $this->_count > 0 condition for hiding attributes with 0 count
		// if ( 'list' == $args['style'] && $this->_count > 0 && $this->_tax_flag ) {
		if ( 'list' == $args['style'] && $this->_tax_flag ) {
			$output .= "\t<li";
			$css_classes = array(
				'cat-item',
				'cat-item-' . $category->term_id,
			);

			$arg = 'filter_' . str_replace( 'pa_', '', $taxonomy );

			$current_filter = ( isset( $_GET[ $arg ] ) ) ? explode( ',', $_GET[ $arg ] ) : array();

			if ( in_array( $category->slug, $current_filter ) ) {
				$css_classes[] = 'chosen';
			}

			if ( ! empty( $tax_terms ) ) {
				foreach( $tax_terms as $tax_term ) {
					if ( $tax_term == $category->slug ) {
						$css_classes[] = 'chosen';
					}
				}
			}

			$css_classes = implode( ' ', apply_filters( 'category_css_class', $css_classes, $category, $depth, $args ) );

			$output .=  ' class="' . $css_classes . '"';
			$output .= ">$link";
		} else {
			return;
		}
	}

	public function end_el( &$output, $category, $depth = 0, $args = array() ) {

		if ( 'list' != $args['style'] || $this->_count < 1 ) {
			return;
		}
		$output .= "</li>\n";
	}
}

/**
 * Ajax loadmore data function
 * Generates data for use in JS file
 */
function sswcaf_ajax_loadmore_data() {

		global $wp_query;

		$options = get_option( 'sswcaf_settings' );
		if ( isset( $options ) ) {
			$disable_ajax_loadmore = isset( $options['sswcaf_disable_loadmore'] ) ? $options['sswcaf_disable_loadmore'] : false;
		}
		if ( ! $disable_ajax_loadmore ) {
			$max		= $wp_query->max_num_pages;
			$paged		= max( 1, $wp_query->get( 'paged' ) );
			$per_page	= $wp_query->get( 'posts_per_page' );
			$total		= $wp_query->found_posts;
			$first		= ( $per_page * $paged ) - $per_page + 1;
			$last		= min( $total, $wp_query->get( 'posts_per_page' ) * $paged );

			if ( $total <= $per_page || -1 == $per_page ) {
				$result_text = _x( 'Showing all %d results', '%d = total', 'sswcaf' );
			} else {
				$result_text = _x( 'Showing %1$d&ndash;%2$d of %3$d results', '%1$d = first, %2$d = last, %3$d = total', 'sswcaf' );
			}

			$localization = array(
				'startPage'		=> $paged,
				'maxPages'		=> $max,
				'perPage'		=> $per_page,
				'total'			=> $total,
				'resultText'	=> $result_text
			);

			$encoded = json_encode($localization);
			echo '<input type="hidden" id="ajax-loadmore-data" data-localization=\'' . $encoded . '\'>';
		}
		else {
			return;
		}
}

add_action( 'woocommerce_after_shop_loop', 'sswcaf_ajax_loadmore_data' );


/**
 * Layered Nav Init
 */
function sswcaf_layered_nav_init() {

	if ( is_active_widget( false, false, 'ss_product_attribute_filters', true ) && ! is_admin() ) {

		global $_chosen_attributes;

		$_chosen_attributes = array();

		$attribute_taxonomies = wc_get_attribute_taxonomies();
		if ( $attribute_taxonomies ) {
			foreach ( $attribute_taxonomies as $tax ) {
				$attribute       = wc_sanitize_taxonomy_name( $tax->attribute_name );
				$taxonomy        = wc_attribute_taxonomy_name( $attribute );
				$name            = 'filter_' . $attribute;
				$query_type_name = 'query_type_' . $attribute;

				if ( ! empty( $_GET[ $name ] ) && taxonomy_exists( $taxonomy ) ) {

					$_chosen_attributes[ $taxonomy ]['terms'] = explode( ',', $_GET[ $name ] );

					if ( empty( $_GET[ $query_type_name ] ) || ! in_array( strtolower( $_GET[ $query_type_name ] ), array( 'and', 'or' ) ) )
						$_chosen_attributes[ $taxonomy ]['query_type'] = apply_filters( 'woocommerce_layered_nav_default_query_type', 'and' );
					else
						$_chosen_attributes[ $taxonomy ]['query_type'] = strtolower( $_GET[ $query_type_name ] );

				}
			}
		}

		add_filter('loop_shop_post_in', 'sswcaf_layered_nav_query' );
	}
}


/**
 * Layered Nav post filter
 *
 * @param array $filtered_posts
 * @return array
 */
function sswcaf_layered_nav_query( $filtered_posts ) {
	global $_chosen_attributes, $wc_query;
	$wc_query = new WC_Query();

	if ( sizeof( $_chosen_attributes ) > 0 ) {

		$matched_products   = array(
			'and' => array(),
			'or'  => array()
		);

		$filtered_attribute = array(
			'and' => false,
			'or'  => false
		);

		foreach ( $_chosen_attributes as $attribute => $data ) {
			$matched_products_from_attribute = array();
			$filtered = false;

			if ( sizeof( $data['terms'] ) > 0 ) {
				foreach ( $data['terms'] as $value ) {

					$posts = get_posts(
						array(
							'post_type' 	=> 'product',
							'numberposts' 	=> -1,
							'post_status' 	=> 'publish',
							'fields' 		=> 'ids',
							'no_found_rows' => true,
							'tax_query' => array(
								array(
									'taxonomy' 	=> $attribute,
									'terms' 	=> $value,
									'field' 	=> 'term_id'
								)
							)
						)
					);

					if ( ! is_wp_error( $posts ) ) {

						if ( sizeof( $matched_products_from_attribute ) > 0 || $filtered ) {
							$matched_products_from_attribute = $data['query_type'] == 'or' ? array_merge( $posts, $matched_products_from_attribute ) : array_intersect( $posts, $matched_products_from_attribute );
						} else {
							$matched_products_from_attribute = $posts;
						}

						$filtered = true;
					}
				}
			}

			if ( sizeof( $matched_products[ $data['query_type'] ] ) > 0 || $filtered_attribute[ $data['query_type'] ] === true ) {
				$matched_products[ $data['query_type'] ] = ( $data['query_type'] == 'or' ) ? array_merge( $matched_products_from_attribute, $matched_products[ $data['query_type'] ] ) : array_intersect( $matched_products_from_attribute, $matched_products[ $data['query_type'] ] );
			} else {
				$matched_products[ $data['query_type'] ] = $matched_products_from_attribute;
			}

			$filtered_attribute[ $data['query_type'] ] = true;

			$wc_query->filtered_product_ids_for_taxonomy[ $attribute ] = $matched_products_from_attribute;
		}

		// Combine our AND and OR result sets
		if ( $filtered_attribute['and'] && $filtered_attribute['or'] )
			$results = array_intersect( $matched_products[ 'and' ], $matched_products[ 'or' ] );
		else
			$results = array_merge( $matched_products[ 'and' ], $matched_products[ 'or' ] );

		if ( $filtered ) {

			WC()->query->layered_nav_post__in   = $results;
			WC()->query->layered_nav_post__in[] = 0;

			if ( sizeof( $filtered_posts ) == 0 ) {
				$filtered_posts   = $results;
				$filtered_posts[] = 0;
			} else {
				$filtered_posts   = array_intersect( $filtered_posts, $results );
				$filtered_posts[] = 0;
			}

		}
	}
	return (array) $filtered_posts;
}

/**
 * Price filter Init
 */
function sswcaf_price_filter_init() {
	if ( is_active_widget( false, false, 'ss_product_price_range', true ) && ! is_admin() ) {
		add_filter( 'loop_shop_post_in', 'sswcaf_price_filter' );
	}
}

/**
 * Price Filter post filter
 *
 * @param array $filtered_posts
 * @return array
 */
function sswcaf_price_filter( $filtered_posts = array() ) {
	global $wpdb;

	if ( isset( $_GET['max_price'] ) || isset( $_GET['min_price'] ) ) {

		$matched_products = array();
		$min              = isset( $_GET['min_price'] ) ? floatval( $_GET['min_price'] ) : 0;
		$max              = isset( $_GET['max_price'] ) ? floatval( $_GET['max_price'] ) : 9999999999;

		// If displaying prices in the shop including taxes, but prices don't include taxes..
		if ( wc_tax_enabled() && 'incl' === get_option( 'woocommerce_tax_display_shop' ) && ! wc_prices_include_tax() ) {
			$tax_classes = array_merge( array( '' ), WC_Tax::get_tax_classes() );

			foreach ( $tax_classes as $tax_class ) {
				$tax_rates = WC_Tax::get_rates( $tax_class );
				$min_class = $min - WC_Tax::get_tax_total( WC_Tax::calc_inclusive_tax( $min, $tax_rates ) );
				$max_class = $max - WC_Tax::get_tax_total( WC_Tax::calc_inclusive_tax( $max, $tax_rates ) );

				$matched_products_query = apply_filters( 'woocommerce_price_filter_results', $wpdb->get_results( $wpdb->prepare( "
					SELECT DISTINCT ID, post_parent, post_type FROM {$wpdb->posts}
					INNER JOIN {$wpdb->postmeta} pm1 ON ID = pm1.post_id
					INNER JOIN {$wpdb->postmeta} pm2 ON ID = pm2.post_id
					WHERE post_type IN ( 'product', 'product_variation' )
					AND post_status = 'publish'
					AND pm1.meta_key IN ('" . implode( "','", array_map( 'esc_sql', apply_filters( 'woocommerce_price_filter_meta_keys', array( '_price' ) ) ) ) . "')
					AND pm1.meta_value BETWEEN %d AND %d
					AND pm2.meta_key = '_tax_class'
					AND pm2.meta_value = %s
				", $min_class, $max_class, sanitize_title( $tax_class ) ), OBJECT_K ), $min_class, $max_class );

				if ( $matched_products_query ) {
					foreach ( $matched_products_query as $product ) {
						if ( $product->post_type == 'product' ) {
							$matched_products[] = $product->ID;
						}
						if ( $product->post_parent > 0 ) {
							$matched_products[] = $product->post_parent;
						}
					}
				}
			}
		} else {
			$matched_products_query = apply_filters( 'woocommerce_price_filter_results', $wpdb->get_results( $wpdb->prepare( "
				SELECT DISTINCT ID, post_parent, post_type FROM {$wpdb->posts}
				INNER JOIN {$wpdb->postmeta} pm1 ON ID = pm1.post_id
				WHERE post_type IN ( 'product', 'product_variation' )
				AND post_status = 'publish'
				AND pm1.meta_key IN ('" . implode( "','", array_map( 'esc_sql', apply_filters( 'woocommerce_price_filter_meta_keys', array( '_price' ) ) ) ) . "')
				AND pm1.meta_value BETWEEN %d AND %d
			", $min, $max ), OBJECT_K ), $min, $max );

			if ( $matched_products_query ) {
				foreach ( $matched_products_query as $product ) {
					if ( $product->post_type == 'product' ) {
						$matched_products[] = $product->ID;
					}
					if ( $product->post_parent > 0 ) {
						$matched_products[] = $product->post_parent;
					}
				}
			}
		}

		$matched_products = array_unique( $matched_products );

		// Filter the id's
		if ( 0 === sizeof( $filtered_posts ) ) {
			$filtered_posts = $matched_products;
		} else {
			$filtered_posts = array_intersect( $filtered_posts, $matched_products );
		}
		$filtered_posts[] = 0;
	}

	return (array) $filtered_posts;
}

//add_action( 'init', 'sswcaf_layered_nav_init' );
//add_action( 'init', 'sswcaf_price_filter_init' );

/**
 * Count products for price range
 */
function sswcaf_price_count( $min = 0, $max = 9999999999 ) {
	if ( (int)$max <= 0 ) {
		$max = 9999999999;
	}
	$query = array(
		'post_status' => 'publish',
		'post_type' => 'product',
		'posts_per_page' => -1,
		'meta_query' => array(
			array(
				'key' => '_price',
				'value' => array( (int)$min, (int)$max ),
				'compare' => 'BETWEEN',
				'type' => 'NUMERIC'
			)
		)
	);

	$result = new WP_Query( $query );
	$count = $result->post_count;
	if ( isset( $count ) && $count >= 0 ) {
		return $count;
	}
	else {
		return 0;
	}
}


/**
 * Show admin notice if WooCommerce
 * plugin is not installed
 */

function sswcaf_install_woocommerce_notice() {
	if ( ! class_exists( 'woocommerce' ) ) {
		echo sprintf( '<div class="error"><p>%s</p></div>', esc_attr__( 'SS WooCommerce Ajax Filters requires WooCommerce plugin to work correctly. Kindly install and activate WooCommerce plugin.', 'sswcaf' ) );
	}
	else {
		return;
	}
}

add_action('admin_notices', 'sswcaf_install_woocommerce_notice');

/**
 * Add shop sidebar toggle button for mobile in footer
 */
function sswcaf_shop_sidebar_actions() {

	$options = get_option( 'sswcaf_settings' );
	$sswcaf_select_sidebar = isset( $options['sswcaf_select_sidebar'] ) ? $options['sswcaf_select_sidebar'] : null;

	if ( ( is_post_type_archive( 'product' ) || is_tax( get_object_taxonomies( 'product' ) ) ) && is_active_sidebar( $sswcaf_select_sidebar ) ) : ?>

        <div id="sliding-panel-actions">
            <div  class="container clearfix">
                <?php echo apply_filters( 'sswcaf_shop_sidebar_mobile_button', sprintf( '<a class="panel-toggle" href="#">%s</a>', esc_html__( 'Filters', 'sswcaf' ) ) ); ?>
            </div><!-- /.container -->
        </div><!-- /#sliding-panel-actions -->

        <div class="panel-body-mask"></div><!-- /.panel-body-mask -->

	<?php endif;

}

add_action( 'wp_footer', 'sswcaf_shop_sidebar_actions' );
?>